<?php
require_once '../config/config.php';

if (!isAdmin()) {
    redirect('../dashboard.php');
}

$db = Database::getInstance()->getConnection();

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create') {
        $username = sanitize($_POST['username'] ?? '');
        $nama_lengkap = sanitize($_POST['nama_lengkap'] ?? '');
        $email = sanitize($_POST['email'] ?? '');
        $no_hp = sanitize($_POST['no_hp'] ?? '');
        $no_ktp = sanitize($_POST['no_ktp'] ?? '');
        $role = sanitize($_POST['role'] ?? 'warga');
        $password = $_POST['password'] ?? '';
        
        // For petugas_tangki, use depo_id; for others, use regular address fields
        $depo_id = null;
        $provinsi = '';
        $kabupaten = '';
        $kelurahan = '';
        $dusun_id = null;
        $rt = '';
        $rw = '';
        
        if ($role === 'petugas_tangki') {
            $depo_id = intval($_POST['depo_id'] ?? 0);
            if ($depo_id <= 0) {
                $_SESSION['error'] = 'Depo harus dipilih untuk petugas tangki';
                redirect('users.php');
            }
            // Get depo address and use it as kelurahan (alamat lengkap)
            $stmt = $db->prepare("SELECT alamat, nama_depo FROM depo_pengiriman WHERE id = ?");
            $stmt->execute([$depo_id]);
            $depo = $stmt->fetch();
            if ($depo) {
                // Use depo alamat as kelurahan field for display purposes
                $kelurahan = !empty($depo['alamat']) ? $depo['alamat'] : $depo['nama_depo'];
                $provinsi = '';
                $kabupaten = '';
                $dusun_id = null;
                $rt = '';
                $rw = '';
            } else {
                $_SESSION['error'] = 'Depo tidak ditemukan';
                redirect('users.php');
            }
        } else {
            $provinsi = sanitize($_POST['provinsi'] ?? '');
            $kabupaten = sanitize($_POST['kabupaten'] ?? '');
            $kelurahan = sanitize($_POST['kelurahan'] ?? '');
            $dusun_id = intval($_POST['dusun_id'] ?? 0);
            $rt = sanitize($_POST['rt'] ?? '');
            $rw = sanitize($_POST['rw'] ?? '');
        }
        
        if (empty($username) || empty($nama_lengkap) || empty($password)) {
            $_SESSION['error'] = 'Data wajib harus diisi';
        } else {
            // Check username
            $stmt = $db->prepare("SELECT id FROM users WHERE username = ?");
            $stmt->execute([$username]);
            if ($stmt->fetch()) {
                $_SESSION['error'] = 'Username sudah digunakan';
            } elseif (!empty($no_ktp)) {
                // Check KTP
                $stmt = $db->prepare("SELECT id FROM users WHERE no_ktp = ?");
                $stmt->execute([$no_ktp]);
                if ($stmt->fetch()) {
                    $_SESSION['error'] = 'Nomor KTP sudah terdaftar';
                } else {
                    $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                    $stmt = $db->prepare("INSERT INTO users (username, password, nama_lengkap, email, no_hp, no_ktp, provinsi, kabupaten, kelurahan, dusun_id, rt, rw, role, depo_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                    if ($stmt->execute([$username, $hashed_password, $nama_lengkap, $email, $no_hp, $no_ktp, $provinsi, $kabupaten, $kelurahan, $dusun_id > 0 ? $dusun_id : null, $rt, $rw, $role, $depo_id > 0 ? $depo_id : null])) {
                        $_SESSION['success'] = 'Akun berhasil dibuat';
                    } else {
                        $_SESSION['error'] = 'Gagal membuat akun';
                    }
                }
            } else {
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                $stmt = $db->prepare("INSERT INTO users (username, password, nama_lengkap, email, no_hp, no_ktp, provinsi, kabupaten, kelurahan, dusun_id, rt, rw, role, depo_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                if ($stmt->execute([$username, $hashed_password, $nama_lengkap, $email, $no_hp, $no_ktp, $provinsi, $kabupaten, $kelurahan, $dusun_id > 0 ? $dusun_id : null, $rt, $rw, $role, $depo_id > 0 ? $depo_id : null])) {
                    $_SESSION['success'] = 'Akun berhasil dibuat';
                } else {
                    $_SESSION['error'] = 'Gagal membuat akun';
                }
            }
        }
    } elseif ($action === 'reset_password') {
        $user_id = intval($_POST['user_id'] ?? 0);
        $new_password = $_POST['new_password'] ?? '';
        
        if (empty($new_password) || strlen($new_password) < 6) {
            $_SESSION['error'] = 'Password minimal 6 karakter';
        } else {
            $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
            $stmt = $db->prepare("UPDATE users SET password = ? WHERE id = ?");
            if ($stmt->execute([$hashed_password, $user_id])) {
                $_SESSION['success'] = 'Password berhasil direset';
            } else {
                $_SESSION['error'] = 'Gagal reset password';
            }
        }
    } elseif ($action === 'change_password') {
        $user_id = intval($_POST['user_id'] ?? 0);
        $new_password = $_POST['new_password'] ?? '';
        
        if (empty($new_password) || strlen($new_password) < 6) {
            $_SESSION['error'] = 'Password minimal 6 karakter';
        } else {
            $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
            $stmt = $db->prepare("UPDATE users SET password = ? WHERE id = ?");
            if ($stmt->execute([$hashed_password, $user_id])) {
                $_SESSION['success'] = 'Password berhasil diubah';
            } else {
                $_SESSION['error'] = 'Gagal mengubah password';
            }
        }
    } elseif ($action === 'reset_all_passwords') {
        $new_password = $_POST['new_password'] ?? '';
        $role_filter = sanitize($_POST['role_filter'] ?? '');
        
        if (empty($new_password) || strlen($new_password) < 6) {
            $_SESSION['error'] = 'Password minimal 6 karakter';
        } else {
            $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
            
            if (!empty($role_filter)) {
                $stmt = $db->prepare("UPDATE users SET password = ? WHERE role = ? AND id != ?");
                $result = $stmt->execute([$hashed_password, $role_filter, $_SESSION['user_id']]);
                $affected = $stmt->rowCount();
            } else {
                $stmt = $db->prepare("UPDATE users SET password = ? WHERE id != ?");
                $result = $stmt->execute([$hashed_password, $_SESSION['user_id']]);
                $affected = $stmt->rowCount();
            }
            
            if ($result) {
                $_SESSION['success'] = "Password berhasil direset untuk $affected pengguna";
            } else {
                $_SESSION['error'] = 'Gagal reset password';
            }
        }
    } elseif ($action === 'update') {
        $user_id = intval($_POST['user_id'] ?? 0);
        $username = sanitize($_POST['username'] ?? '');
        $nama_lengkap = sanitize($_POST['nama_lengkap'] ?? '');
        $email = sanitize($_POST['email'] ?? '');
        $no_hp = sanitize($_POST['no_hp'] ?? '');
        $no_ktp = sanitize($_POST['no_ktp'] ?? '');
        $role = sanitize($_POST['role'] ?? 'warga');
        $status = sanitize($_POST['status'] ?? 'active');
        
        // For petugas_tangki, use depo_id; for others, use regular address fields
        $depo_id = null;
        $provinsi = '';
        $kabupaten = '';
        $kelurahan = '';
        $dusun_id = null;
        $rt = '';
        $rw = '';
        
        if ($role === 'petugas_tangki') {
            $depo_id = intval($_POST['depo_id'] ?? 0);
            if ($depo_id <= 0) {
                $_SESSION['error'] = 'Depo harus dipilih untuk petugas tangki';
                redirect('users.php');
            }
            // Get depo address and use it as kelurahan (alamat lengkap)
            $stmt = $db->prepare("SELECT alamat, nama_depo FROM depo_pengiriman WHERE id = ?");
            $stmt->execute([$depo_id]);
            $depo = $stmt->fetch();
            if ($depo) {
                // Use depo alamat as kelurahan field for display purposes
                $kelurahan = !empty($depo['alamat']) ? $depo['alamat'] : $depo['nama_depo'];
                $provinsi = '';
                $kabupaten = '';
                $dusun_id = null;
                $rt = '';
                $rw = '';
            } else {
                $_SESSION['error'] = 'Depo tidak ditemukan';
                redirect('users.php');
            }
        } else {
            $provinsi = sanitize($_POST['provinsi'] ?? '');
            $kabupaten = sanitize($_POST['kabupaten'] ?? '');
            $kelurahan = sanitize($_POST['kelurahan'] ?? '');
            $dusun_id = intval($_POST['dusun_id'] ?? 0);
            $rt = sanitize($_POST['rt'] ?? '');
            $rw = sanitize($_POST['rw'] ?? '');
        }
        
        // Check KTP if changed
        if (!empty($no_ktp)) {
            $stmt = $db->prepare("SELECT id FROM users WHERE no_ktp = ? AND id != ?");
            $stmt->execute([$no_ktp, $user_id]);
            if ($stmt->fetch()) {
                $_SESSION['error'] = 'Nomor KTP sudah terdaftar';
                redirect('users.php');
            }
        }
        
        $stmt = $db->prepare("UPDATE users SET username = ?, nama_lengkap = ?, email = ?, no_hp = ?, no_ktp = ?, provinsi = ?, kabupaten = ?, kelurahan = ?, dusun_id = ?, rt = ?, rw = ?, role = ?, status = ?, depo_id = ? WHERE id = ?");
        if ($stmt->execute([$username, $nama_lengkap, $email, $no_hp, $no_ktp, $provinsi, $kabupaten, $kelurahan, $dusun_id > 0 ? $dusun_id : null, $rt, $rw, $role, $status, $depo_id > 0 ? $depo_id : null, $user_id])) {
            $_SESSION['success'] = 'Data berhasil diupdate';
        } else {
            $_SESSION['error'] = 'Gagal update data';
        }
    } elseif ($action === 'delete') {
        $user_id = intval($_POST['user_id'] ?? 0);
        $stmt = $db->prepare("DELETE FROM users WHERE id = ? AND id != ?");
        if ($stmt->execute([$user_id, $_SESSION['user_id']])) {
            $_SESSION['success'] = 'Akun berhasil dihapus';
        } else {
            $_SESSION['error'] = 'Gagal menghapus akun';
        }
    }
    
    redirect('users.php');
}

// Get all users with dusun info and depo info
$stmt = $db->query("SELECT u.*, d.nama_dusun, des.nama_desa, dep.nama_depo, dep.alamat as depo_alamat
                    FROM users u 
                    LEFT JOIN dusun d ON u.dusun_id = d.id 
                    LEFT JOIN desa des ON d.desa_id = des.id 
                    LEFT JOIN depo_pengiriman dep ON u.depo_id = dep.id
                    ORDER BY u.created_at DESC");
$users = $stmt->fetchAll();

// Get dusun list for dropdown
$stmt = $db->query("SELECT d.*, des.nama_desa FROM dusun d JOIN desa des ON d.desa_id = des.id ORDER BY des.nama_desa, d.nama_dusun");
$dusun_list = $stmt->fetchAll();

// Get depo list for dropdown
$stmt = $db->query("SELECT * FROM depo_pengiriman WHERE is_active = 1 ORDER BY nama_depo");
$depo_list = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kelola Pengguna - <?php echo APP_NAME; ?></title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <aside class="sidebar">
        <div class="sidebar-header">
            <h2><i class="fas fa-tint"></i> <?php echo APP_NAME; ?></h2>
        </div>
        <nav class="sidebar-nav">
            <a href="../dashboard.php" class="nav-item">
                <i class="fas fa-home"></i>
                <span>Dashboard</span>
            </a>
            <a href="users.php" class="nav-item active">
                <i class="fas fa-users"></i>
                <span>Kelola Pengguna</span>
            </a>
            <a href="desa.php" class="nav-item">
                <i class="fas fa-building"></i>
                <span>Kelola Desa</span>
            </a>
            <a href="menu_icons.php" class="nav-item">
                <i class="fas fa-palette"></i>
                <span>Kelola Menu</span>
            </a>
            <a href="../auth/logout.php" class="nav-item">
                <i class="fas fa-sign-out-alt"></i>
                <span>Logout</span>
            </a>
        </nav>
    </aside>
    
    <main class="main-content">
        <header class="content-header">
            <div class="header-left">
                <a href="../dashboard.php" class="btn btn-secondary" style="margin-bottom: 10px; display: inline-flex; align-items: center; gap: 8px; text-decoration: none;">
                    <i class="fas fa-arrow-left"></i> Kembali ke Dashboard
                </a>
                <h1><i class="fas fa-users"></i> Kelola Pengguna</h1>
                <p>Manajemen akun pengguna sistem</p>
            </div>
            <div style="display: flex; gap: 10px; flex-wrap: wrap;">
                <button class="btn btn-primary" onclick="document.getElementById('createModal').style.display='block'">
                    <i class="fas fa-user-plus"></i> Tambah Pengguna
                </button>
                <button class="btn btn-warning" onclick="document.getElementById('resetAllModal').style.display='block'">
                    <i class="fas fa-key"></i> Reset Password Semua
                </button>
            </div>
        </header>

        <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i> <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
        </div>
        <?php endif; ?>

        <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger">
            <i class="fas fa-exclamation-circle"></i> <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
        </div>
        <?php endif; ?>

        <div class="content-section">
            <div class="table-responsive">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Username</th>
                            <th>Nama Lengkap</th>
                            <th>No. KTP</th>
                            <th>Alamat</th>
                            <th>Email</th>
                            <th>No. HP</th>
                            <th>Role</th>
                            <th>Status</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users as $user): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($user['username']); ?></td>
                            <td><?php echo htmlspecialchars($user['nama_lengkap']); ?></td>
                            <td><?php echo htmlspecialchars($user['no_ktp'] ?? '-'); ?></td>
                            <td>
                                <?php 
                                if ($user['role'] === 'petugas_tangki' && !empty($user['nama_depo'])) {
                                    echo '<i class="fas fa-warehouse"></i> ' . htmlspecialchars($user['nama_depo']);
                                    if (!empty($user['depo_alamat'])) {
                                        echo '<br><small style="color: #666;">' . htmlspecialchars($user['depo_alamat']) . '</small>';
                                    }
                                } else {
                                    $alamat_parts = [];
                                    if (!empty($user['rt'])) $alamat_parts[] = 'RT ' . $user['rt'];
                                    if (!empty($user['rw'])) $alamat_parts[] = 'RW ' . $user['rw'];
                                    if (!empty($user['nama_dusun'])) $alamat_parts[] = $user['nama_dusun'];
                                    if (!empty($user['kelurahan'])) $alamat_parts[] = $user['kelurahan'];
                                    if (!empty($user['kabupaten'])) $alamat_parts[] = $user['kabupaten'];
                                    echo htmlspecialchars(implode(', ', $alamat_parts) ?: '-');
                                }
                                ?>
                            </td>
                            <td><?php echo htmlspecialchars($user['email'] ?? '-'); ?></td>
                            <td><?php echo htmlspecialchars($user['no_hp'] ?? '-'); ?></td>
                            <td>
                                <span class="badge badge-info"><?php echo ucfirst($user['role']); ?></span>
                            </td>
                            <td>
                                <span class="badge badge-<?php echo $user['status'] === 'active' ? 'success' : 'danger'; ?>">
                                    <?php echo ucfirst($user['status']); ?>
                                </span>
                            </td>
                            <td>
                                <button class="btn btn-sm btn-primary" onclick="editUser(<?php echo htmlspecialchars(json_encode($user)); ?>)">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button class="btn btn-sm btn-warning btn-reset-password" 
                                        data-user-id="<?php echo $user['id']; ?>" 
                                        data-username="<?php echo htmlspecialchars($user['username']); ?>">
                                    <i class="fas fa-key"></i> Reset
                                </button>
                                <button class="btn btn-sm btn-info btn-change-password" 
                                        data-user-id="<?php echo $user['id']; ?>" 
                                        data-username="<?php echo htmlspecialchars($user['username']); ?>">
                                    <i class="fas fa-lock"></i> Ubah
                                </button>
                                <?php if ($user['id'] != $_SESSION['user_id']): ?>
                                <button class="btn btn-sm btn-danger" onclick="deleteUser(<?php echo $user['id']; ?>)">
                                    <i class="fas fa-trash"></i>
                                </button>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </main>

    <!-- Create Modal -->
    <div id="createModal" class="modal" style="display:none;">
        <div class="modal-content">
            <span class="close" onclick="document.getElementById('createModal').style.display='none'">&times;</span>
            <h2>Tambah Pengguna</h2>
            <form method="POST" class="form-container">
                <input type="hidden" name="action" value="create">
                <div class="form-group">
                    <label>Username</label>
                    <input type="text" name="username" required>
                </div>
                <div class="form-group">
                    <label>Nama Lengkap</label>
                    <input type="text" name="nama_lengkap" required>
                </div>
                <div class="form-group">
                    <label>Nomor KTP</label>
                    <input type="text" name="no_ktp" maxlength="16" pattern="[0-9]{16}" placeholder="16 digit angka">
                </div>
                <!-- Address fields for non-petugas_tangki -->
                <div id="addressFields" class="address-fields">
                    <div class="form-row">
                        <div class="form-group">
                            <label>Provinsi</label>
                            <input type="text" name="provinsi" id="create_provinsi" placeholder="Contoh: DI Yogyakarta">
                        </div>
                        <div class="form-group">
                            <label>Kabupaten</label>
                            <input type="text" name="kabupaten" id="create_kabupaten" placeholder="Contoh: Gunungkidul">
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group">
                            <label>Kelurahan</label>
                            <input type="text" name="kelurahan" id="create_kelurahan" placeholder="Nama Kelurahan">
                        </div>
                        <div class="form-group">
                            <label>Dusun</label>
                            <select name="dusun_id" id="create_dusun_id">
                                <option value="">Pilih Dusun</option>
                                <?php foreach ($dusun_list as $dusun): ?>
                                <option value="<?php echo $dusun['id']; ?>"><?php echo htmlspecialchars($dusun['nama_desa']); ?> - <?php echo htmlspecialchars($dusun['nama_dusun']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group">
                            <label>RT</label>
                            <input type="text" name="rt" id="create_rt" placeholder="Contoh: 01" maxlength="10">
                        </div>
                        <div class="form-group">
                            <label>RW</label>
                            <input type="text" name="rw" id="create_rw" placeholder="Contoh: 05" maxlength="10">
                        </div>
                    </div>
                </div>
                
                <!-- Depo field for petugas_tangki -->
                <div id="depoField" class="depo-field" style="display: none;">
                    <div class="form-group">
                        <label><i class="fas fa-warehouse"></i> Depo</label>
                        <select name="depo_id" id="create_depo_id" required onchange="showDepoAddress(this, 'create')">
                            <option value="">Pilih Depo</option>
                            <?php foreach ($depo_list as $depo): ?>
                            <option value="<?php echo $depo['id']; ?>" 
                                    data-alamat="<?php echo htmlspecialchars($depo['alamat'] ?? ''); ?>"
                                    data-nama="<?php echo htmlspecialchars($depo['nama_depo']); ?>">
                                <?php echo htmlspecialchars($depo['nama_depo']); ?> - <?php echo htmlspecialchars($depo['alamat'] ?? ''); ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                        <div id="create_depo_address_display" style="margin-top: 10px; padding: 10px; background: #f0f0f0; border-radius: 5px; display: none;">
                            <strong><i class="fas fa-map-marker-alt"></i> Alamat Depo:</strong>
                            <p id="create_depo_address_text" style="margin: 5px 0 0 0; color: #333;"></p>
                        </div>
                        <small style="color: #666; display: block; margin-top: 5px;">
                            <i class="fas fa-info-circle"></i> Pilih depo tempat petugas tangki bertugas. Alamat depo akan digunakan sebagai alamat petugas.
                        </small>
                    </div>
                </div>
                <div class="form-group">
                    <label>Email</label>
                    <input type="email" name="email">
                </div>
                <div class="form-group">
                    <label>No. HP</label>
                    <input type="text" name="no_hp" placeholder="Contoh: 081234567890">
                </div>
                <div class="form-group">
                    <label>Role</label>
                    <select name="role" id="create_role" required onchange="toggleAddressFields(this.value, 'create')">
                        <option value="warga">Warga</option>
                        <option value="pemdes">Pemdes</option>
                        <option value="petugas_tangki">Petugas Tangki</option>
                        <option value="admin">Admin</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Password</label>
                    <input type="password" name="password" required minlength="6">
                </div>
                <button type="submit" class="btn btn-primary btn-block">Simpan</button>
            </form>
        </div>
    </div>

    <script src="../assets/js/main.js"></script>
    <script>
    // Make functions globally available
    window.showResetPasswordModal = function(userId, username) {
        const modal = document.getElementById('resetPasswordModal');
        const userIdInput = document.getElementById('reset_user_id');
        const usernameSpan = document.getElementById('reset_username');
        
        if (modal && userIdInput && usernameSpan) {
            userIdInput.value = userId;
            usernameSpan.textContent = username;
            // Reset form
            const form = modal.querySelector('form');
            if (form) {
                form.reset();
                userIdInput.value = userId; // Set again after reset
            }
            modal.style.display = 'block';
        } else {
            console.error('Modal elements not found');
            alert('Error: Modal tidak ditemukan. Silakan refresh halaman.');
        }
    };
    
    window.showChangePasswordModal = function(userId, username) {
        const modal = document.getElementById('changePasswordModal');
        const userIdInput = document.getElementById('change_user_id');
        const usernameSpan = document.getElementById('change_username');
        
        if (modal && userIdInput && usernameSpan) {
            userIdInput.value = userId;
            usernameSpan.textContent = username;
            // Reset form
            const form = modal.querySelector('form');
            if (form) {
                form.reset();
                userIdInput.value = userId; // Set again after reset
            }
            modal.style.display = 'block';
        } else {
            console.error('Modal elements not found');
            alert('Error: Modal tidak ditemukan. Silakan refresh halaman.');
        }
    };
    
    // Toggle address fields based on role
    function toggleAddressFields(role, prefix) {
        const addressFields = document.getElementById(prefix === 'create' ? 'addressFields' : 'editAddressFields');
        const depoField = document.getElementById(prefix === 'create' ? 'depoField' : 'editDepoField');
        const depoDisplay = document.getElementById(prefix + '_depo_address_display');
        const depoSelect = document.getElementById(prefix === 'create' ? 'create_depo_id' : 'edit_depo_id');
        
        if (role === 'petugas_tangki') {
            if (addressFields) addressFields.style.display = 'none';
            if (depoField) depoField.style.display = 'block';
        } else {
            if (addressFields) addressFields.style.display = 'block';
            if (depoField) depoField.style.display = 'none';
            // Hide depo address display and reset depo select when role is not petugas_tangki
            if (depoDisplay) depoDisplay.style.display = 'none';
            if (depoSelect) depoSelect.value = '';
        }
    }
    
    // Show depo address when depo is selected
    function showDepoAddress(selectElement, prefix) {
        const selectedOption = selectElement.options[selectElement.selectedIndex];
        const alamat = selectedOption.getAttribute('data-alamat') || '';
        const nama = selectedOption.getAttribute('data-nama') || '';
        
        const displayDiv = document.getElementById(prefix + '_depo_address_display');
        const addressText = document.getElementById(prefix + '_depo_address_text');
        
        if (displayDiv && addressText) {
            if (selectElement.value && alamat) {
                addressText.textContent = alamat;
                displayDiv.style.display = 'block';
            } else {
                displayDiv.style.display = 'none';
            }
        }
    }
    
    function editUser(user) {
        document.getElementById('editModal').style.display = 'block';
        document.getElementById('edit_user_id').value = user.id;
        document.getElementById('edit_username').value = user.username;
        document.getElementById('edit_nama_lengkap').value = user.nama_lengkap;
        document.getElementById('edit_no_ktp').value = user.no_ktp || '';
        document.getElementById('edit_provinsi').value = user.provinsi || '';
        document.getElementById('edit_kabupaten').value = user.kabupaten || '';
        document.getElementById('edit_kelurahan').value = user.kelurahan || '';
        document.getElementById('edit_dusun_id').value = user.dusun_id || '';
        document.getElementById('edit_rt').value = user.rt || '';
        document.getElementById('edit_rw').value = user.rw || '';
        document.getElementById('edit_email').value = user.email || '';
        document.getElementById('edit_no_hp').value = user.no_hp || '';
        document.getElementById('edit_role').value = user.role;
        document.getElementById('edit_status').value = user.status;
        
        // Toggle fields based on role first
        toggleAddressFields(user.role, 'edit');
        
        // Handle depo_id for petugas_tangki
        if (user.role === 'petugas_tangki' && user.depo_id) {
            const depoSelect = document.getElementById('edit_depo_id');
            if (depoSelect) {
                depoSelect.value = user.depo_id;
                // Trigger change to show depo address after a short delay to ensure DOM is ready
                setTimeout(function() {
                    showDepoAddress(depoSelect, 'edit');
                }, 100);
            }
        } else if (user.role === 'petugas_tangki') {
            // If petugas_tangki but no depo_id, hide address display
            const depoDisplay = document.getElementById('edit_depo_address_display');
            if (depoDisplay) depoDisplay.style.display = 'none';
        }
    }
    
    function deleteUser(userId) {
        if (confirm('Hapus user ini?')) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="user_id" value="${userId}">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }
    </script>
    
    <!-- Edit Modal -->
    <div id="editModal" class="modal" style="display:none;">
        <div class="modal-content">
            <span class="close" onclick="document.getElementById('editModal').style.display='none'">&times;</span>
            <h2>Edit Pengguna</h2>
            <form method="POST" class="form-container">
                <input type="hidden" name="action" value="update">
                <input type="hidden" name="user_id" id="edit_user_id">
                <div class="form-group">
                    <label>Username</label>
                    <input type="text" name="username" id="edit_username" required>
                </div>
                <div class="form-group">
                    <label>Nama Lengkap</label>
                    <input type="text" name="nama_lengkap" id="edit_nama_lengkap" required>
                </div>
                <div class="form-group">
                    <label>Nomor KTP</label>
                    <input type="text" name="no_ktp" id="edit_no_ktp" maxlength="16" pattern="[0-9]{16}" placeholder="16 digit angka">
                </div>
                <!-- Address fields for non-petugas_tangki -->
                <div id="editAddressFields" class="address-fields">
                    <div class="form-row">
                        <div class="form-group">
                            <label>Provinsi</label>
                            <input type="text" name="provinsi" id="edit_provinsi" placeholder="Contoh: DI Yogyakarta">
                        </div>
                        <div class="form-group">
                            <label>Kabupaten</label>
                            <input type="text" name="kabupaten" id="edit_kabupaten" placeholder="Contoh: Gunungkidul">
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group">
                            <label>Kelurahan</label>
                            <input type="text" name="kelurahan" id="edit_kelurahan" placeholder="Nama Kelurahan">
                        </div>
                        <div class="form-group">
                            <label>Dusun</label>
                            <select name="dusun_id" id="edit_dusun_id">
                                <option value="">Pilih Dusun</option>
                                <?php foreach ($dusun_list as $dusun): ?>
                                <option value="<?php echo $dusun['id']; ?>"><?php echo htmlspecialchars($dusun['nama_desa']); ?> - <?php echo htmlspecialchars($dusun['nama_dusun']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group">
                            <label>RT</label>
                            <input type="text" name="rt" id="edit_rt" placeholder="Contoh: 01" maxlength="10">
                        </div>
                        <div class="form-group">
                            <label>RW</label>
                            <input type="text" name="rw" id="edit_rw" placeholder="Contoh: 05" maxlength="10">
                        </div>
                    </div>
                </div>
                
                <!-- Depo field for petugas_tangki -->
                <div id="editDepoField" class="depo-field" style="display: none;">
                    <div class="form-group">
                        <label><i class="fas fa-warehouse"></i> Depo</label>
                        <select name="depo_id" id="edit_depo_id" required onchange="showDepoAddress(this, 'edit')">
                            <option value="">Pilih Depo</option>
                            <?php foreach ($depo_list as $depo): ?>
                            <option value="<?php echo $depo['id']; ?>" 
                                    data-alamat="<?php echo htmlspecialchars($depo['alamat'] ?? ''); ?>"
                                    data-nama="<?php echo htmlspecialchars($depo['nama_depo']); ?>">
                                <?php echo htmlspecialchars($depo['nama_depo']); ?> - <?php echo htmlspecialchars($depo['alamat'] ?? ''); ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                        <div id="edit_depo_address_display" style="margin-top: 10px; padding: 10px; background: #f0f0f0; border-radius: 5px; display: none;">
                            <strong><i class="fas fa-map-marker-alt"></i> Alamat Depo:</strong>
                            <p id="edit_depo_address_text" style="margin: 5px 0 0 0; color: #333;"></p>
                        </div>
                        <small style="color: #666; display: block; margin-top: 5px;">
                            <i class="fas fa-info-circle"></i> Pilih depo tempat petugas tangki bertugas. Alamat depo akan digunakan sebagai alamat petugas.
                        </small>
                    </div>
                </div>
                <div class="form-group">
                    <label>Email</label>
                    <input type="email" name="email" id="edit_email">
                </div>
                <div class="form-group">
                    <label>No. HP</label>
                    <input type="text" name="no_hp" id="edit_no_hp" placeholder="Contoh: 081234567890">
                </div>
                <div class="form-group">
                    <label>Role</label>
                    <select name="role" id="edit_role" required onchange="toggleAddressFields(this.value, 'edit')">
                        <option value="warga">Warga</option>
                        <option value="pemdes">Pemdes</option>
                        <option value="petugas_tangki">Petugas Tangki</option>
                        <option value="admin">Admin</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Status</label>
                    <select name="status" id="edit_status" required>
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                    </select>
                </div>
                <button type="submit" class="btn btn-primary btn-block">Update</button>
            </form>
        </div>
    </div>
    
    <style>
    .modal {
        display: none;
        position: fixed;
        z-index: 1000;
        left: 0;
        top: 0;
        width: 100%;
        height: 100%;
        background: rgba(0,0,0,0.5);
    }
    
    .modal-content {
        background: white;
        margin: 5% auto;
        padding: 30px;
        border-radius: 15px;
        width: 90%;
        max-width: 600px;
        max-height: 90vh;
        overflow-y: auto;
    }
    
    .close {
        float: right;
        font-size: 28px;
        font-weight: bold;
        cursor: pointer;
        color: #999;
        line-height: 20px;
    }
    
    .close:hover {
        color: #333;
    }
    
    /* Ensure buttons are clickable */
    .btn, .btn-sm, button {
        pointer-events: auto !important;
        cursor: pointer !important;
        z-index: 10 !important;
        position: relative !important;
        touch-action: manipulation !important;
        -webkit-tap-highlight-color: rgba(0, 0, 0, 0.1) !important;
    }
    
    /* Ensure table buttons are clickable */
    .data-table .btn,
    .data-table .btn-sm,
    .data-table button {
        pointer-events: auto !important;
        cursor: pointer !important;
        z-index: 10 !important;
        position: relative !important;
    }
    
    /* Prevent overlay from blocking buttons */
    .modal {
        z-index: 1000 !important;
    }
    
    .modal-content {
        z-index: 1001 !important;
        position: relative !important;
    }
    </style>
    
    <!-- Reset Password Modal -->
    <div id="resetPasswordModal" class="modal" style="display:none;">
        <div class="modal-content">
            <span class="close" onclick="document.getElementById('resetPasswordModal').style.display='none'">&times;</span>
            <h2><i class="fas fa-key"></i> Reset Password</h2>
            <form method="POST" class="form-container">
                <input type="hidden" name="action" value="reset_password">
                <input type="hidden" name="user_id" id="reset_user_id">
                <div class="alert alert-info">
                    <i class="fas fa-info-circle"></i> Reset password untuk user: <strong id="reset_username"></strong>
                </div>
                <div class="form-group">
                    <label>Password Baru</label>
                    <input type="password" name="new_password" id="reset_new_password" required minlength="6" placeholder="Masukkan password baru">
                    <small style="color: #666;">Minimal 6 karakter</small>
                </div>
                <div class="form-group">
                    <label>Konfirmasi Password</label>
                    <input type="password" id="reset_confirm_password" required minlength="6" placeholder="Konfirmasi password baru">
                </div>
                <div class="form-group">
                    <button type="submit" class="btn btn-warning btn-block">
                        <i class="fas fa-key"></i> Reset Password
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Change Password Modal -->
    <div id="changePasswordModal" class="modal" style="display:none;">
        <div class="modal-content">
            <span class="close" onclick="document.getElementById('changePasswordModal').style.display='none'">&times;</span>
            <h2><i class="fas fa-lock"></i> Ubah Password</h2>
            <form method="POST" class="form-container">
                <input type="hidden" name="action" value="change_password">
                <input type="hidden" name="user_id" id="change_user_id">
                <div class="alert alert-info">
                    <i class="fas fa-info-circle"></i> Ubah password untuk user: <strong id="change_username"></strong>
                </div>
                <div class="form-group">
                    <label>Password Baru</label>
                    <input type="password" name="new_password" id="change_new_password" required minlength="6" placeholder="Masukkan password baru">
                    <small style="color: #666;">Minimal 6 karakter</small>
                </div>
                <div class="form-group">
                    <label>Konfirmasi Password</label>
                    <input type="password" id="change_confirm_password" required minlength="6" placeholder="Konfirmasi password baru">
                </div>
                <div class="form-group">
                    <button type="submit" class="btn btn-primary btn-block">
                        <i class="fas fa-lock"></i> Ubah Password
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Reset All Passwords Modal -->
    <div id="resetAllModal" class="modal" style="display:none;">
        <div class="modal-content">
            <span class="close" onclick="document.getElementById('resetAllModal').style.display='none'">&times;</span>
            <h2><i class="fas fa-key"></i> Reset Password Semua Pengguna</h2>
            <form method="POST" class="form-container" onsubmit="return confirm('Yakin ingin reset password semua pengguna? Tindakan ini tidak dapat dibatalkan!');">
                <input type="hidden" name="action" value="reset_all_passwords">
                <div class="alert alert-warning">
                    <i class="fas fa-exclamation-triangle"></i> 
                    <strong>Peringatan:</strong> Tindakan ini akan mengubah password semua pengguna (kecuali admin yang sedang login) menjadi password yang sama!
                </div>
                <div class="form-group">
                    <label>Filter Berdasarkan Role (Opsional)</label>
                    <select name="role_filter" id="role_filter">
                        <option value="">Semua Role</option>
                        <option value="warga">Warga</option>
                        <option value="pemdes">Pemdes</option>
                        <option value="petugas_tangki">Petugas Tangki</option>
                        <option value="admin">Admin</option>
                    </select>
                    <small style="color: #666;">Kosongkan untuk reset semua pengguna</small>
                </div>
                <div class="form-group">
                    <label>Password Baru untuk Semua</label>
                    <input type="password" name="new_password" id="reset_all_password" required minlength="6" placeholder="Masukkan password baru">
                    <small style="color: #666;">Minimal 6 karakter. Password ini akan diterapkan ke semua pengguna yang dipilih.</small>
                </div>
                <div class="form-group">
                    <label>Konfirmasi Password</label>
                    <input type="password" id="reset_all_confirm_password" required minlength="6" placeholder="Konfirmasi password baru">
                </div>
                <div class="form-group">
                    <button type="submit" class="btn btn-warning btn-block">
                        <i class="fas fa-key"></i> Reset Password Semua
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <script>
    // Password confirmation validation
    document.addEventListener('DOMContentLoaded', function() {
        // Reset password form
        const resetForm = document.querySelector('#resetPasswordModal form');
        if (resetForm) {
            resetForm.addEventListener('submit', function(e) {
                const password = document.getElementById('reset_new_password').value;
                const confirm = document.getElementById('reset_confirm_password').value;
                if (password !== confirm) {
                    e.preventDefault();
                    alert('Password dan konfirmasi password tidak sama!');
                    return false;
                }
            });
        }
        
        // Change password form
        const changeForm = document.querySelector('#changePasswordModal form');
        if (changeForm) {
            changeForm.addEventListener('submit', function(e) {
                const password = document.getElementById('change_new_password').value;
                const confirm = document.getElementById('change_confirm_password').value;
                if (password !== confirm) {
                    e.preventDefault();
                    alert('Password dan konfirmasi password tidak sama!');
                    return false;
                }
            });
        }
        
        // Reset all passwords form
        const resetAllForm = document.querySelector('#resetAllModal form');
        if (resetAllForm) {
            resetAllForm.addEventListener('submit', function(e) {
                const password = document.getElementById('reset_all_password').value;
                const confirm = document.getElementById('reset_all_confirm_password').value;
                if (password !== confirm) {
                    e.preventDefault();
                    alert('Password dan konfirmasi password tidak sama!');
                    return false;
                }
            });
        }
    });
    
    // Close modal when clicking outside
    window.addEventListener('click', function(event) {
        const modals = document.querySelectorAll('.modal');
        modals.forEach(modal => {
            if (event.target == modal) {
                modal.style.display = 'none';
            }
        });
    });
    
    // Ensure buttons work on mobile and desktop
    document.addEventListener('DOMContentLoaded', function() {
        // Initialize role toggle on page load
        const createRole = document.getElementById('create_role');
        if (createRole) {
            toggleAddressFields(createRole.value, 'create');
        }
        
        // Add event listener for create depo select
        const createDepoSelect = document.getElementById('create_depo_id');
        if (createDepoSelect) {
            createDepoSelect.addEventListener('change', function() {
                showDepoAddress(this, 'create');
            });
        }
        
        // Add event listener for edit depo select
        const editDepoSelect = document.getElementById('edit_depo_id');
        if (editDepoSelect) {
            editDepoSelect.addEventListener('change', function() {
                showDepoAddress(this, 'edit');
            });
        }
        
        // Reset form when create modal is opened
        const createModal = document.getElementById('createModal');
        if (createModal) {
            const createForm = createModal.querySelector('form');
            if (createForm) {
                createForm.addEventListener('reset', function() {
                    setTimeout(function() {
                        const role = document.getElementById('create_role');
                        if (role) {
                            toggleAddressFields(role.value, 'create');
                        }
                        // Hide depo address display on reset
                        const depoDisplay = document.getElementById('create_depo_address_display');
                        if (depoDisplay) depoDisplay.style.display = 'none';
                    }, 10);
                });
            }
        }
        // Make sure all buttons are clickable
        const buttons = document.querySelectorAll('.btn, .btn-sm, button');
        buttons.forEach(btn => {
            btn.style.pointerEvents = 'auto';
            btn.style.cursor = 'pointer';
            btn.style.zIndex = '10';
            btn.style.position = 'relative';
        });
        
        // Use data attributes for reset password buttons
        document.querySelectorAll('.btn-reset-password').forEach(btn => {
            btn.addEventListener('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                const userId = this.getAttribute('data-user-id');
                const username = this.getAttribute('data-username');
                if (userId && username) {
                    showResetPasswordModal(parseInt(userId), username);
                } else {
                    console.error('Missing data attributes');
                }
            });
        });
        
        // Use data attributes for change password buttons
        document.querySelectorAll('.btn-change-password').forEach(btn => {
            btn.addEventListener('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                const userId = this.getAttribute('data-user-id');
                const username = this.getAttribute('data-username');
                if (userId && username) {
                    showChangePasswordModal(parseInt(userId), username);
                } else {
                    console.error('Missing data attributes');
                }
            });
        });
        
        // Debug: Log if modals exist
        console.log('Reset Modal:', document.getElementById('resetPasswordModal'));
        console.log('Change Modal:', document.getElementById('changePasswordModal'));
    });
    </script>
</body>
</html>

