-- Database: sumberai
-- SumberAI - Peringatan Dini Kekeringan & Manajemen Air Desa

CREATE DATABASE IF NOT EXISTS sumberai CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE sumberai;

-- Table: users
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    nama_lengkap VARCHAR(100) NOT NULL,
    email VARCHAR(100),
    no_hp VARCHAR(20),
    no_ktp VARCHAR(20),
    provinsi VARCHAR(100),
    kabupaten VARCHAR(100),
    kelurahan VARCHAR(100),
    dusun_id INT,
    rt VARCHAR(10),
    rw VARCHAR(10),
    latitude DECIMAL(10, 8),
    longitude DECIMAL(11, 8),
    role ENUM('admin', 'pemdes', 'petugas_tangki', 'warga') DEFAULT 'warga',
    status ENUM('active', 'inactive') DEFAULT 'active',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (dusun_id) REFERENCES dusun(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Table: desa
CREATE TABLE IF NOT EXISTS desa (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nama_desa VARCHAR(100) NOT NULL,
    kode_desa VARCHAR(20) UNIQUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Table: dusun
CREATE TABLE IF NOT EXISTS dusun (
    id INT AUTO_INCREMENT PRIMARY KEY,
    desa_id INT NOT NULL,
    nama_dusun VARCHAR(100) NOT NULL,
    rt VARCHAR(10),
    rw VARCHAR(10),
    alamat TEXT,
    latitude DECIMAL(10, 8),
    longitude DECIMAL(11, 8),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (desa_id) REFERENCES desa(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Table: data_curah_hujan
CREATE TABLE IF NOT EXISTS data_curah_hujan (
    id INT AUTO_INCREMENT PRIMARY KEY,
    dusun_id INT NOT NULL,
    tanggal DATE NOT NULL,
    curah_hujan DECIMAL(5, 2) DEFAULT 0,
    kelembapan_tanah DECIMAL(5, 2),
    suhu DECIMAL(5, 2),
    sumber_data ENUM('bmkg', 'sensor', 'manual') DEFAULT 'manual',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (dusun_id) REFERENCES dusun(id) ON DELETE CASCADE,
    INDEX idx_tanggal (tanggal)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Table: laporan_warga
CREATE TABLE IF NOT EXISTS laporan_warga (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    dusun_id INT NOT NULL,
    tanggal DATE NOT NULL,
    jenis_laporan ENUM('kekeringan', 'ketersediaan_air', 'kualitas_air', 'penampungan', 'sumur', 'pam', 'lainnya') NOT NULL,
    sumber_air ENUM('penampungan', 'sumur', 'pam', 'lainnya') DEFAULT 'lainnya',
    deskripsi TEXT,
    foto VARCHAR(255),
    latitude DECIMAL(10, 8),
    longitude DECIMAL(11, 8),
    status ENUM('pending', 'verified', 'rejected') DEFAULT 'pending',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (dusun_id) REFERENCES dusun(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Table: prediksi_kekeringan
CREATE TABLE IF NOT EXISTS prediksi_kekeringan (
    id INT AUTO_INCREMENT PRIMARY KEY,
    dusun_id INT NOT NULL,
    tanggal_prediksi DATE NOT NULL,
    tingkat_risiko ENUM('aman', 'waspada', 'kering') NOT NULL,
    skor_risiko DECIMAL(5, 2),
    prediksi_hari INT DEFAULT 7,
    faktor_utama TEXT,
    rekomendasi TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (dusun_id) REFERENCES dusun(id) ON DELETE CASCADE,
    INDEX idx_tanggal (tanggal_prediksi)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Table: depo_pengiriman
CREATE TABLE IF NOT EXISTS depo_pengiriman (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nama_depo VARCHAR(100) NOT NULL,
    alamat TEXT,
    latitude DECIMAL(10, 8) NOT NULL,
    longitude DECIMAL(11, 8) NOT NULL,
    kapasitas_liter INT DEFAULT 0,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Table: distribusi_tangki
CREATE TABLE IF NOT EXISTS distribusi_tangki (
    id INT AUTO_INCREMENT PRIMARY KEY,
    dusun_id INT NOT NULL,
    tanggal_jadwal DATE NOT NULL,
    status ENUM('menunggu_approval', 'disetujui', 'ditolak', 'penyiapan', 'dalam_perjalanan', 'selesai', 'dibatalkan') DEFAULT 'menunggu_approval',
    volume_liter INT,
    petugas_id INT,
    depo_id INT,
    jarak_km DECIMAL(8, 2),
    estimasi_bbm_liter DECIMAL(8, 2),
    biaya_bbm DECIMAL(10, 2),
    catatan TEXT,
    alasan_reject TEXT,
    foto_bukti VARCHAR(255),
    latitude_bukti DECIMAL(10, 8),
    longitude_bukti DECIMAL(11, 8),
    approved_by INT,
    approved_at TIMESTAMP NULL,
    urutan_rute INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (dusun_id) REFERENCES dusun(id) ON DELETE CASCADE,
    FOREIGN KEY (petugas_id) REFERENCES users(id) ON DELETE SET NULL,
    FOREIGN KEY (depo_id) REFERENCES depo_pengiriman(id) ON DELETE SET NULL,
    FOREIGN KEY (approved_by) REFERENCES users(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Table: notifikasi
CREATE TABLE IF NOT EXISTS notifikasi (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT,
    judul VARCHAR(200) NOT NULL,
    pesan TEXT NOT NULL,
    tipe ENUM('info', 'warning', 'danger', 'success') DEFAULT 'info',
    is_read TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_read (user_id, is_read)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Table: menu_icons
CREATE TABLE IF NOT EXISTS menu_icons (
    id INT AUTO_INCREMENT PRIMARY KEY,
    menu_key VARCHAR(50) UNIQUE NOT NULL,
    menu_label VARCHAR(100) NOT NULL,
    icon_class VARCHAR(100) NOT NULL,
    icon_color VARCHAR(20) DEFAULT '#3498db',
    urutan INT DEFAULT 0,
    is_active TINYINT(1) DEFAULT 1,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Table: wa_config
CREATE TABLE IF NOT EXISTS wa_config (
    id INT AUTO_INCREMENT PRIMARY KEY,
    api_url VARCHAR(255) NOT NULL,
    api_key VARCHAR(255),
    api_secret VARCHAR(255),
    sender_number VARCHAR(20),
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Table: wa_broadcast
CREATE TABLE IF NOT EXISTS wa_broadcast (
    id INT AUTO_INCREMENT PRIMARY KEY,
    judul VARCHAR(200) NOT NULL,
    pesan TEXT NOT NULL,
    gambar VARCHAR(255),
    filter_desa_id INT,
    filter_dusun_id INT,
    filter_status VARCHAR(20),
    total_target INT DEFAULT 0,
    total_terkirim INT DEFAULT 0,
    total_gagal INT DEFAULT 0,
    status ENUM('pending', 'sending', 'completed', 'failed') DEFAULT 'pending',
    created_by INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (filter_desa_id) REFERENCES desa(id) ON DELETE SET NULL,
    FOREIGN KEY (filter_dusun_id) REFERENCES dusun(id) ON DELETE SET NULL,
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Table: wa_broadcast_log
CREATE TABLE IF NOT EXISTS wa_broadcast_log (
    id INT AUTO_INCREMENT PRIMARY KEY,
    broadcast_id INT NOT NULL,
    user_id INT NOT NULL,
    phone_number VARCHAR(20) NOT NULL,
    status ENUM('success', 'failed') DEFAULT 'failed',
    error_message TEXT,
    sent_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (broadcast_id) REFERENCES wa_broadcast(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Table: app_config
CREATE TABLE IF NOT EXISTS app_config (
    id INT AUTO_INCREMENT PRIMARY KEY,
    config_key VARCHAR(50) UNIQUE NOT NULL,
    config_value TEXT,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Table: role_permissions
CREATE TABLE IF NOT EXISTS role_permissions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    role VARCHAR(50) NOT NULL,
    permission_key VARCHAR(100) NOT NULL,
    is_allowed TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY unique_role_permission (role, permission_key)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert default app config
INSERT INTO app_config (config_key, config_value) VALUES
('app_name', 'SumberAI'),
('app_desc', 'Peringatan Dini Kekeringan & Manajemen Air Desa'),
('app_logo', ''),
('app_favicon', '')
ON DUPLICATE KEY UPDATE config_value = VALUES(config_value);

-- Insert default role permissions
INSERT INTO role_permissions (role, permission_key, is_allowed) VALUES
('admin', 'all', 1),
('pemdes', 'view_prediksi', 1),
('pemdes', 'view_laporan', 1),
('pemdes', 'manage_distribusi', 1),
('pemdes', 'view_notifikasi', 1),
('warga', 'view_prediksi', 1),
('warga', 'view_peta', 1),
('warga', 'create_laporan', 1),
('warga', 'view_distribusi', 1),
('warga', 'view_notifikasi', 1),
('petugas_tangki', 'view_distribusi', 1),
('petugas_tangki', 'update_distribusi', 1),
('petugas_tangki', 'view_notifikasi', 1)
ON DUPLICATE KEY UPDATE is_allowed = VALUES(is_allowed);

-- Insert default admin (password: admin123)
INSERT INTO users (username, password, nama_lengkap, email, role) VALUES
('admin', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'Administrator', 'admin@sumberai.com', 'admin');

-- Insert default menu icons
INSERT INTO menu_icons (menu_key, menu_label, icon_class, icon_color, urutan) VALUES
('dashboard', 'Dashboard', 'fas fa-home', '#3498db', 1),
('prediksi', 'Prediksi Kekeringan', 'fas fa-chart-line', '#e74c3c', 2),
('peta', 'Peta Desa', 'fas fa-map-marked-alt', '#2ecc71', 3),
('laporan', 'Laporan Warga', 'fas fa-file-alt', '#f39c12', 4),
('distribusi', 'Distribusi Tangki', 'fas fa-truck', '#9b59b6', 5),
('optimasi', 'AI Optimasi Pengiriman', 'fas fa-route', '#e91e63', 6),
('desa', 'Kelola Desa', 'fas fa-building', '#1abc9c', 7),
('users', 'Kelola Pengguna', 'fas fa-users', '#34495e', 8),
('depo', 'Kelola Depo', 'fas fa-warehouse', '#ff9800', 9),
('notifikasi', 'Notifikasi', 'fas fa-bell', '#e67e22', 10),
('laporan_sistem', 'Laporan Sistem', 'fas fa-chart-bar', '#16a085', 11),
('pengaturan', 'Pengaturan', 'fas fa-cog', '#7f8c8d', 12),
('menu_icons', 'Kelola Menu', 'fas fa-palette', '#9c27b0', 13),
('wa_broadcast', 'Broadcast WA', 'fas fa-whatsapp', '#25D366', 14),
('hak_akses', 'Hak Akses', 'fas fa-user-shield', '#6c757d', 15);

