// SumberAI - Main JavaScript

// Tab switching
document.addEventListener('DOMContentLoaded', function() {
    const tabButtons = document.querySelectorAll('.tab-btn');
    const tabContents = document.querySelectorAll('.tab-content');
    
    tabButtons.forEach(button => {
        button.addEventListener('click', () => {
            const targetTab = button.getAttribute('data-tab');
            
            // Remove active class from all
            tabButtons.forEach(btn => btn.classList.remove('active'));
            tabContents.forEach(content => content.classList.remove('active'));
            
            // Add active class to clicked
            button.classList.add('active');
            document.getElementById(targetTab + '-tab').classList.add('active');
        });
    });
    
    // Notification dropdown
    const notifBtn = document.getElementById('notifBtn');
    const notifMenu = document.getElementById('notifMenu');
    
    if (notifBtn && notifMenu) {
        notifBtn.addEventListener('click', (e) => {
            e.stopPropagation();
            notifMenu.classList.toggle('active');
        });
        
        document.addEventListener('click', (e) => {
            if (!notifBtn.contains(e.target) && !notifMenu.contains(e.target)) {
                notifMenu.classList.remove('active');
            }
        });
    }
    
    // Mobile menu toggle - Pastikan berfungsi untuk semua role
    const sidebar = document.querySelector('.sidebar') || document.getElementById('sidebar');
    const menuToggle = document.querySelector('.menu-toggle') || document.getElementById('menuToggle');
    
    // Create overlay for mobile
    let overlay = document.querySelector('.sidebar-overlay');
    if (!overlay && sidebar) {
        overlay = document.createElement('div');
        overlay.className = 'sidebar-overlay';
        document.body.appendChild(overlay);
    }
    
    // Toggle sidebar function
    function toggleSidebar() {
        const sidebarEl = document.querySelector('.sidebar') || document.getElementById('sidebar');
        if (sidebarEl) {
            // Pastikan sidebar visible di mobile
            if (window.innerWidth <= 768) {
                sidebarEl.style.display = 'block';
                sidebarEl.style.visibility = 'visible';
                sidebarEl.style.opacity = '1';
                sidebarEl.style.transform = sidebarEl.classList.contains('active') ? 'translateX(0)' : 'translateX(-100%)';
            }
            sidebarEl.classList.toggle('active');
            
            // Update transform directly for immediate feedback
            if (sidebarEl.classList.contains('active')) {
                sidebarEl.style.transform = 'translateX(0)';
            } else {
                sidebarEl.style.transform = 'translateX(-100%)';
            }
            
            if (overlay) {
                overlay.classList.toggle('active');
            }
            // Prevent body scroll when sidebar is open
            if (sidebarEl.classList.contains('active')) {
                document.body.style.overflow = 'hidden';
            } else {
                document.body.style.overflow = '';
            }
        }
    }
    
    // Make toggleSidebar globally available
    window.toggleSidebar = toggleSidebar;
    
    if (menuToggle) {
        // Pastikan button bisa diklik
        menuToggle.style.pointerEvents = 'auto';
        menuToggle.style.cursor = 'pointer';
        menuToggle.style.zIndex = '1002';
        menuToggle.style.position = 'fixed';
        menuToggle.style.display = 'flex';
        
        // Remove any existing onclick to avoid conflicts
        menuToggle.removeAttribute('onclick');
        
        // Click event - use capture phase for better mobile support
        menuToggle.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            e.stopImmediatePropagation();
            toggleSidebar();
            return false;
        }, true);
        
        // Touch event untuk mobile
        menuToggle.addEventListener('touchend', function(e) {
            e.preventDefault();
            e.stopPropagation();
            e.stopImmediatePropagation();
            toggleSidebar();
            return false;
        }, true);
        
        // Mouse down as fallback
        menuToggle.addEventListener('mousedown', function(e) {
            e.preventDefault();
            toggleSidebar();
        });
    }
    
    // Close sidebar when clicking overlay
    if (overlay) {
        overlay.addEventListener('click', () => {
            toggleSidebar();
        });
    }
    
    // Close sidebar when clicking on nav item on mobile
    const navItems = document.querySelectorAll('.sidebar-nav .nav-item');
    navItems.forEach(item => {
        item.addEventListener('click', () => {
            if (window.innerWidth <= 768) {
                setTimeout(() => {
                    toggleSidebar();
                }, 100);
            }
        });
    });
    
    // Close sidebar when clicking outside on mobile
    document.addEventListener('click', (e) => {
        if (window.innerWidth <= 768 && sidebar && sidebar.classList.contains('active')) {
            if (!sidebar.contains(e.target) && !menuToggle.contains(e.target)) {
                toggleSidebar();
            }
        }
    });
    
    // Auto-hide alerts
    const alerts = document.querySelectorAll('.alert');
    alerts.forEach(alert => {
        setTimeout(() => {
            alert.style.transition = 'opacity 0.5s';
            alert.style.opacity = '0';
            setTimeout(() => alert.remove(), 500);
        }, 5000);
    });
    
    // Form validation
    const forms = document.querySelectorAll('form');
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            const requiredFields = form.querySelectorAll('[required]');
            let isValid = true;
            
            requiredFields.forEach(field => {
                if (!field.value.trim()) {
                    isValid = false;
                    field.style.borderColor = '#e74c3c';
                } else {
                    field.style.borderColor = '';
                }
            });
            
            if (!isValid) {
                e.preventDefault();
                alert('Mohon lengkapi semua field yang wajib diisi');
            }
        });
    });
});

// Map initialization (Leaflet)
function initMap(lat = -7.7956, lng = 110.3695) {
    if (typeof L !== 'undefined') {
        const map = L.map('map').setView([lat, lng], 13);
        
        L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
            attribution: '© OpenStreetMap contributors'
        }).addTo(map);
        
        return map;
    }
    return null;
}

// AJAX helper
function ajaxRequest(url, method = 'GET', data = null) {
    return new Promise((resolve, reject) => {
        const xhr = new XMLHttpRequest();
        xhr.open(method, url, true);
        xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
        
        xhr.onload = function() {
            if (xhr.status >= 200 && xhr.status < 300) {
                try {
                    const response = JSON.parse(xhr.responseText);
                    resolve(response);
                } catch (e) {
                    resolve(xhr.responseText);
                }
            } else {
                reject(new Error('Request failed'));
            }
        };
        
        xhr.onerror = () => reject(new Error('Network error'));
        
        if (data) {
            xhr.send(new URLSearchParams(data).toString());
        } else {
            xhr.send();
        }
    });
}

// Show loading
function showLoading(element) {
    element.innerHTML = '<div class="loading"></div>';
}

// Format number
function formatNumber(num) {
    return new Intl.NumberFormat('id-ID').format(num);
}

// Show toast notification
function showToast(message, type = 'info') {
    const toast = document.createElement('div');
    toast.className = `alert alert-${type}`;
    toast.textContent = message;
    toast.style.position = 'fixed';
    toast.style.zIndex = '9999';
    toast.style.minWidth = '250px';
    toast.style.maxWidth = '90%';
    toast.style.wordWrap = 'break-word';
    
    // Mobile responsive positioning
    if (window.innerWidth <= 768) {
        toast.style.top = '70px';
        toast.style.left = '50%';
        toast.style.transform = 'translateX(-50%)';
        toast.style.right = 'auto';
        toast.style.minWidth = 'calc(100% - 30px)';
        toast.style.maxWidth = 'calc(100% - 30px)';
    } else {
        toast.style.top = '20px';
        toast.style.right = '20px';
    }
    
    document.body.appendChild(toast);
    
    setTimeout(() => {
        toast.style.transition = 'opacity 0.5s';
        toast.style.opacity = '0';
        setTimeout(() => toast.remove(), 500);
    }, 3000);
}

